package com.worldline.information

import android.annotation.SuppressLint
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.os.Bundle
import android.os.Handler
import android.os.IBinder
import android.os.Looper
import android.os.Message
import android.os.Messenger
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.core.os.bundleOf
import java.util.UUID

class MainActivity : ComponentActivity() {
    /** Messenger for communicating with the service.  */
    private var mService: Messenger? = null

    /** Flag indicating whether we have called bind on the service.  */
    private var bound: Boolean = false
    private lateinit var updateJson: (String) -> Unit

    /**
     * Class for interacting with the main interface of the service.
     */
    private val mConnection = object : ServiceConnection {
        // Called when the connection with the service is established.
        override fun onServiceConnected(className: ComponentName, service: IBinder) {
            // Because we have bound to an explicit
            // service that is running in our own process, we can
            // cast its IBinder to a concrete class and directly access it.
            mService = Messenger(service)
            bound = true
        }

        // Called when the connection with the service disconnects unexpectedly.
        override fun onServiceDisconnected(className: ComponentName) {
            bound = false
        }
    }

    private val paymentInformationBundle = bundleOf(
        "WPI_SERVICE_TYPE" to "WPI_SVC_PAYMENT_INFORMATION",
        "WPI_VERSION" to "2.1",
        "WPI_SESSION_ID" to UUID.randomUUID().toString()
    )

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContent {
            var paymentInfoJson by remember { mutableStateOf("") }
            updateJson = { paymentInfoJson = it }
            PaymentInfoScreen(paymentInfoJson) {
                executePaymentInformation()
            }
        }
    }

    override fun onStart() {
        super.onStart()
        val intent = Intent("com.worldline.payment.action.PROCESS_INFORMATION").apply {
            component = getWpiSupportedServices("com.worldline.payment.action.PROCESS_INFORMATION").first()
            putExtras(paymentInformationBundle)
        }
        bindService(intent, mConnection, Context.BIND_AUTO_CREATE)
    }

    override fun onStop() {
        super.onStop()
        // Unbind from the service.
        if (bound) {
            unbindService(mConnection)
            bound = false
        }
    }

    @SuppressLint("HandlerLeak", "UnsafeIntentLaunch")
    private fun executePaymentInformation() {
        val message = Message.obtain(
            null,
            895349,
            paymentInformationBundle
        )
        val handler = object : Handler(Looper.getMainLooper()) {
            override fun handleMessage(msg: Message) {
                when (msg.what) {
                    895349 -> {
                        val data = msg.obj as? Bundle
                        val response = data?.getString("WPI_RESPONSE") ?: "No response"
                        updateJson(response)
                    }
                    else -> throw IllegalArgumentException("${msg.what} is an incorrect message identifier.")
                }
            }
        }
        val messenger = Messenger(handler)
        message.replyTo = messenger
        mService?.send(message)
    }

    private fun Context.getWpiSupportedServices(action: String): List<ComponentName> {
        val intent = Intent(action)
        return packageManager.queryIntentServices(intent, 0)
            .mapNotNull { ComponentName(it.serviceInfo.packageName, it.serviceInfo.name) }
    }
}
